# Cargar librerías necesarias
library(ggplot2)
library(factoextra)
library(cluster)
library(tidyr)

# 1. Crear un dataset con 5 variables y 100 filas
set.seed(123)  # Fijar semilla para reproducibilidad
data <- data.frame(
  Var1 = rnorm(100, mean = 10, sd = 2),
  Var2 = rnorm(100, mean = 20, sd = 3),
  Var3 = rnorm(100, mean = 15, sd = 1.5),
  Var4 = rnorm(100, mean = 5, sd = 1),
  Var5 = rnorm(100, mean = 30, sd = 4)
)

# 2. Calcular los componentes principales (PCA)
pca_result <- prcomp(data, scale = TRUE)  # Escalar las variables para PCA

# 3. Panel de cargas (loadings) de los componentes principales
# Extraer las cargas (loadings) de los componentes principales
loadings <- as.data.frame(pca_result$rotation)  # Las cargas están en pca_result$rotation

# Agregar nombres de las variables y componentes
loadings$Variable <- rownames(loadings)
loadings <- tidyr::gather(loadings, key = "Componente", value = "Carga", -Variable)

# Graficar las cargas como un gráfico de barras
ggplot(loadings, aes(x = Variable, y = Carga, fill = Carga)) +
  geom_bar(stat = "identity", position = "dodge", color = "black") +
  facet_wrap(~ Componente, scales = "free_y") +
  scale_fill_gradient2(low = "#FC4E07", mid = "white", high = "#00AFBB", midpoint = 0) +
  labs(title = "Cargas de los Componentes Principales",
       x = "Variables",
       y = "Carga",
       fill = "Carga") +
  theme_minimal() +
  theme(axis.text.x = element_text(angle = 45, hjust = 1))

# 4. Gráfico de las cargas (loadings) de los componentes principales
fviz_pca_var(pca_result, col.var = "contrib", 
             gradient.cols = c("#00AFBB", "#E7B800", "#FC4E07"),
             repel = TRUE) + 
  ggtitle("Cargas de los Componentes Principales") +
  theme_minimal()

# 5. Mostrar la varianza y la varianza acumulada
varianza <- pca_result$sdev^2  # Varianza de cada componente
varianza_acumulada <- cumsum(varianza) / sum(varianza)  # Varianza acumulada

cat("Varianza de cada componente:\n")
print(varianza)

cat("\nVarianza acumulada:\n")
print(varianza_acumulada)

# 6. Gráfico de barras de la varianza de los componentes (Criterio del codo)
fviz_eig(pca_result, addlabels = TRUE, 
         barfill = "#2E9FDF", barcolor = "#2E9FDF") +
  ggtitle("Varianza de los Componentes Principales") +
  theme_minimal()


# 7. Gráfico de la varianza acumulada con umbral del 80%
varianza_acumulada <- cumsum(pca_result$sdev^2) / sum(pca_result$sdev^2)  # Varianza acumulada
componentes <- 1:length(varianza_acumulada)  # Número de componentes

# Crear un dataframe para el gráfico
varianza_df <- data.frame(
  Componente = componentes,
  Varianza_Acumulada = varianza_acumulada
)

# Graficar la varianza acumulada
ggplot(varianza_df, aes(x = Componente, y = Varianza_Acumulada)) +
  geom_line(color = "#2E9FDF", size = 1.2) +
  geom_point(color = "#FC4E07", size = 3) +
  geom_hline(yintercept = 0.8, color = "red", linetype = "dashed", size = 1) +  # Umbral del 80%
  scale_y_continuous(labels = scales::percent) +  # Formato de porcentaje en el eje y
  labs(title = "Varianza Acumulada de los Componentes Principales",
       x = "Número de Componentes",
       y = "Varianza Acumulada") +
  theme_minimal()
